//------------------------------------------------------------------------------
// Copyright 2008 Mentor Graphics Corporation
// All Rights Reserved Worldwide
// 
// Licensed under the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.  You may obtain
// a copy of the License at
// 
//        http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
// License for the specific language governing permissions and limitations
// under the License.
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Title: OVM Sequences
//
// This file defines the following sequences.
//
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
//
// Group: ovm_apb_rw_sequence
//
// This simple APB transaction sequence generates ~num_trans~ sequence items
// (transactions). The convenience macos `ovm_do_with is not used in order
// that you see how to what the macro does behind the scenes. 
//------------------------------------------------------------------------------

// (begin inline source)
class ovm_apb_rw_sequence extends ovm_sequence #(ovm_apb_item);

  `ovm_object_utils(ovm_apb_rw_sequence)

  rand int unsigned num_trans = 5; 

  constraint max_count { num_trans <= m_sequencer.max_random_count; }

  function new(string name = "apb_random");
    super.new(name);
  endfunction

  task body();

    ovm_apb_tr req = super.req;

    ovm_report_info(get_full_name(), "Write sequence starting");

    for (int i = 0; i < num_trans; i++) begin

      start_item(tr);

      void'(tr.randomize() with {
             addr[9:8] != 2'b11;
             addr[7:0] < 8'd100;
             addr[31:10] == 0;
             data < 8'd100;
          });

      finish_item(tr);

      ovm_report_info(get_type_name(),
         $psprintf("Got response: cmd=%s addr=%h data=%h",
	           req.cmd,req.addr,req.data));
    end
    ovm_report_info(get_full_name(), "Write sequence completing");

  endtask

endclass
// (end inline source)



//------------------------------------------------------------------------------
//
// Group: ovm_apb_rw_sequence_grab
//
// Defines a contrived sequence that exercises the ~grab~ and ~ungrab~ feature.
// A fixed address is used in order to better identify transactions coming from
// this sequence.
//------------------------------------------------------------------------------

// (begin inline source)
class ovm_apb_rw_sequence_grab extends ovm_sequence #(ovm_apb_item);

  `ovm_object_utils(ovm_apb_rw_sequence_grab)

  rand int unsigned fixed_addr;

  function new(string name = "apb_random");
    super.new(name);
  endfunction

  task do_item(ovm_apb_item item, int unsigned _addr, int unsigned _data);
    start_item(item);
    void'(item.randomize() with { addr == _addr; data < _data; });
    finish_item(item);
  endtask

  task body();

    ovm_apb_rw req;

    ovm_report_info(get_full_name(), "Write sequence starting");

    for (int i=0; i<5; i++)
      do_item(req, fixed_addr, 100);

    ovm_report_info("OVM APB Grab Sequence", "grabbing sequencer\n");
    grab();

    // use constant addresses to easily verify grab was successful
    for (int i=0; i<4; i++)
      do_item(req, 4, 100);

    ovm_report_info("OVM APB Grab Sequence", "ungrabbing sequencer\n");
    ungrab();

    for (int i=0; i<5; i++)
      do_item(req, fixed_addr, 100);

    ovm_report_info(get_full_name(), "Write sequence completing");

  endtask


endclass
// (end inline source)


